unit models;

interface

uses sysutils,dialogs;

const maxtriangles = 4096;
      maxvertices  = 2048;
      maxtexco     = 2048;
      maxframes    = 512;
      maxskins     = 32;

Type

   Tvec3 = packed record
      x,y,z : Single;
   end;

   Tmdlheader = packed record
      magic : array [0..3] of char;   // must be idp2
      version : integer;  // quake2 = version 8
      org_x : single; //origin
      org_y : single;
      org_z : single;
      //nono2      : integer;
      frameSize : integer;  //size in byes of one frame
      numSkins : integer;   // number of xxx
      numVertices : integer;
      numTexCos : integer;
      numTriangles : integer;
      numGlCommands : integer;
      numFrames : integer;
      numSurfaces : integer; // byte ofset for xxx
      offsetSkins : integer;
      offsetTexcos : integer;
      offsetTriangles : integer;
      offsetFrames : integer;
      offsetGlCommands: integer;
      offsetSurfaces : integer;
      offsetEnd : integer;
   end;

   TVector3d = packed record
      x,y,z : integer;
   end;

   TTriangleVertex = packed record
      x,y,z : byte;
      lightNormalIndex : byte;
   end;

   Tframe = packed record
      scalex,scaley,scalez : single;  // scale to decode vertexes
      translatex,translatey,translatez : single;// translation to decode vertexes
      name : array [0..15] of char; // name of frame
      vertices : array[0..0]of TtriangleVertex;
   end;

   pframe = ^Tframe;

   TTriangle = packed record
      extra : smallint;
      num_uvframes : smallint;
      vertexIndices : array[0..2] of smallint;
      textureIndices: array[0..2] of smallint;
   end;
   
   TTextureCoordinate = packed record
      s,t : smallint;
   end;

   TskinName = array[0..63] of char;
   TskinNames = array[0..0] of TSkinName;
   TModelInfo = record
      version : integer;
      frameSize : integer;  //size in byes of one frame
      numSkins : integer;   // number of xxx
      //numVertices : integer;
      numTexCos : integer;
      numTriangles : integer;
      numGlCommands : integer;
      numFrames : integer;
      skinwidth,skinheight : integer;
      skinNames : ^TskinNames;
   end;

   Tmodel = record
      Info         : TModelInfo;
      Frames       : ^Tframe;
      SkinsGlIndex : array[0..32] of integer;
      GlCmds       : ^Longint;
      Texcos       : ^TTextureCoordinate;
      Triangles    : ^TTriangle;
   end;

   Pmodel = ^tmodel;

   //Version2 model definitions
   //these stuctures are the ones different with the version 1 structures

   TTriangleVertex2 = packed record
      vert : integer;
      lightNormalIndex : byte;
   end;

   Tframe2 = packed record
      scalex,scaley,scalez : single;            // scale to decode vertexes
      translatex,translatey,translatez : single;// translation to decode vertexes
      name : array [0..15] of char;             // name of frame
      vertices : array[0..0]of TtriangleVertex2;
   end;

   pframe2 = ^Tframe2;

   //Quake 2 model definitions
   //these structures are the ones different with the daikatana ver1 structures

   TQuake2Header = packed record
      magic : array [0..3] of char;   // must be idp2
      version : integer;              // quake2 = version 8
      skinWidth : integer;            // witdth of skins
      skinHeight : integer;           // height of skins
      frameSize : integer;            //size in byes of one frame
      numSkins : integer;             // number of xxx
      numVertices : integer;
      numTexCos : integer;
      numTriangles : integer;
      numGlCommands : integer;
      numFrames : integer;
      offsetSkins : integer;          // byte ofset for xxx
      offsetTexCos : integer;
      offsetTriangles : integer;
      offsetFrames : integer;
      offsetGlCommands : integer;
      offsetEnd : integer;
   end;

   TQuakeTriangle = packed record
      vertexIndices : array[0..2] of smallint;
      textureIndices: array[0..2] of smallint;
   end;

var

userSkin : array[0..63] of char = 'not yet.tga';
userSkinWidth : integer = 256;
userSkinHeight : integer = 256;

procedure ConvertVer2(const filename : string);
procedure ConvertVer1(const filename : string);
function PeekVersion(const FileName : string) : integer;
procedure StringToSkin(const s : string);

procedure dumpheader(const header : TMdlHeader);
procedure dumpheader2(const header : TQuake2Header);

implementation

const
     supportedversions = [1,2];//versions supported by converter
     QuakeXZRescale = 255/2048;
     QuakeYRescale = 255/1024;

function min(x,y : integer) : integer;
begin
     if (x < y) then result := x else result := y;
end;

//Convert the string to an quake representation of the file name
// = 64 chars long and null terminated.
procedure StringToSkin(const s : string);
var
   i : integer;
begin
   // copy chars  
   for i := 0 to min(62,length(s)) do
   begin
        userSkin[i] := s[i+1];
   end;

   // 0 terminated
   userSkin[min(63,length(s)+1)] := #0;
end;

//Decode a version 2 vertex to a 3 component vertex.
function DecodeVertex(int : integer) : TVector3d;
begin
result.z := int and $7FF;
result.y := (int and $1FF800) shr 11;
result.x := (int and $FFE00000) shr 21;
end;

//convert a daitatana triangle record to a quake2 triangle record
//we just ignore the uv_frames
function Dai2QuakeTriangle(const tri : TTriangle) : TQuakeTriangle;
begin
   result.vertexIndices[0] := tri.vertexIndices[0];
   result.vertexIndices[1] := tri.vertexIndices[1];
   result.vertexIndices[2] := tri.vertexIndices[2];

   result.textureIndices[0] := tri.textureIndices[0];
   result.textureIndices[1] := tri.textureIndices[1];
   result.textureIndices[2] := tri.textureIndices[2];

   //writeln(result.vertexIndices[0],' ',result.vertexIndices[1],' ',result.vertexIndices[2]);
end;

//Convert a daikatana vertex to a quake2 vertex
//we loose precision in doeing this
function Dai2QuakeVertex(const vert : TTriangleVertex2) : TTriangleVertex;
var
   v : TVector3d;
begin
   v := DecodeVertex(vert.vert);
   result.x := round(v.x * QuakeXZRescale);
   result.y := round(v.y * QuakeYRescale);
   result.z := round(v.z * QuakeXZRescale);

   result.lightNormalIndex := 1;//vert.lightNormalIndex;
   //writeln(result.x,' ',result.y,' ',result.z,' ',result.lightnormalindex);
end;

//Convert a daikatana frame record to a quake2 frame record
Procedure Dai2QuakeFrame(const frame : TFrame2;var result : TFrame;vertCount : integer);
var
   i : integer;
begin
   result.scalex := frame.scalex/QuakeXZRescale;
   result.scaley := frame.scaley/QuakeYRescale;
   result.scalez := frame.scalez/QuakeXZRescale;
   //riteln(result.scalex,' ',result.scaley,' ',result.scalez);

   //possible incorrect conversion
   result.translatex := frame.translatex;
   result.translatey := frame.translatey;
   result.translatez := frame.translatez;
   //writeln(result.translatex,' ',result.translatey,' ',result.translatez);

   {result.translatex := 1;
   result.translatey := 1;
   result.translatez := 1;
   result.scalex := 1;
   result.scaley := 1;
   result.scalez := 1;}

   for i := 0 to 15 do
      result.name[i] := frame.name[i];

   for i := 0 to vertCount-1 do
      result.vertices[i] := Dai2QuakeVertex(frame.vertices[i]);
end;

type plongint = ^cardinal;

const MAX_GL_SIZE = 1024;
var GlBuff : array [0..(MAX_GL_SIZE*3)-1] of longint;

function Dai2QuakeGlcmds(SrcGlCmds,DstGlcmds : plongint) : integer;
const
   nextitem = 4;
var
   command       : pLongint; // = dword i hope
   dump          : plongint;
   i,num_verts : integer;
   t : single;
   ti : integer;
begin

command := SrcGlCmds;
dump := DstGlCmds;

while true do
 begin
    //read number of vertices
    dump^ := command^;
    num_verts := command^;
    //writeln('vertices: ',num_verts);
    inc(cardinal(command),nextitem);
    inc(cardinal(dump),nextitem);

    //what are these used for?
    //skin indexes??  anyway we skip them for conversion
    inc(cardinal(command),nextitem);
    inc(cardinal(command),nextitem);

    num_verts := abs(num_verts);

    if num_verts = 0 then break;
    if num_verts > 1024 then
    begin
         writeln('To many vertices');
         exit;
    end;
    
    for i := 0 to num_verts-1 do
        begin

        //Daikatana models have reversed order for the glCommand / TexCoords
        //why is unclear but we have to switch them.


        // get vertex index and data
        ti := command^;
        inc(cardinal(command),nextitem);// next 4 bytes integer

        // get & write texture coos
        GlBuff[i*3] := command^;
        inc(cardinal(command),nextitem);// next 4 bytes integer

        GlBuff[i*3+1] := command^;
        inc(cardinal(command),nextitem);// next 4 bytes integer

        //write index
        GlBuff[i*3+2] := ti;
        end;

    for i := 0 to num_verts-1 do
        begin
        Move(glBuff[i*3],dump^,3*nextItem);
        inc(cardinal(dump),3*nextItem);
        end;

    end;//while

    //return new size of glcmds
    result := ((cardinal(dump)-cardinal(DstGlCmds))div 4){+1};
end;

procedure Dai2QuakeHeader(const header : TMdlHeader;var dst : TQuake2Header);
begin
      dst.magic 	:=	      'IDP2';// identify as quake2 model
      dst.version 	:=	      8;//quake2 = version 8
      dst.skinwidth 	:=	      userSkinWidth;//daikatana doesn't store this, quake2 does
      dst.skinheight 	:=	      userSkinHeight;//daikatana doesn't store this, quake2 does
      dst.frameSize 	:=	      header.frameSize;
      dst.numSkins 	:=	      header.numSkins;
      dst.numVertices 	:=	      header.numVertices;
      dst.numTexCos 	:=	      header.numTexCos;
      dst.numTriangles 	:=	      header.numTriangles;
      dst.numGlCommands :=	      header.numGlCommands;
      dst.numFrames 	:=	      header.numFrames;
      //offsets will be regenerated
      dst.offsetSkins 	:=	      header.offsetSkins;
      dst.offsetTexcos 	:=	      header.offsetTexcos;
      dst.offsetTriangles :=	      header.offsetTriangles;
      dst.offsetFrames 	:=	      header.offsetFrames;
      dst.offsetGlCommands:=	      header.offsetGlCommands;
      dst.offsetEnd 	:=	      header.offsetEnd;
end;


(*******************************************************************************

Returns the version of the model in file filename

*******************************************************************************)
function PeekVersion(const FileName : String) : integer;
var
   Header  : TMdlHeader;
   MdlFile : file;
begin
     if not FileExists(FileName) then
     begin
          result := -1;
          exit;
     end;

     AssignFile(MdlFile,FileName);
     Reset(MdlFile,1);
     BlockRead(MdlFile,Header,SizeOf(Header));
     CloseFile(MdlFile);
     result := Header.Version;
end;

(*******************************************************************************

Convert a model of version 1 to a quake 2 model.

*******************************************************************************)
procedure ConvertVer1(const filename : string);
var
   mdlfile,dstfile : File;
   Header  : TMdlHeader;
   QHeader : TQuake2Header;
   Skins   : array[0..9] of Tskinname;
   Frames,src : pFrame;
   i       : integer;
   point   : TVector3d;
   triangles,srctri : ^TTriangle;
   //conversion
   QHearder : TQuake2Header;
   GlCmds,QGlCmds : plongint;
   QFrames,dst : pFrame;
   TexCos : ^TTextureCoordinate;
   QTriangles,dsttri : ^TQuakeTriangle;
begin

   if fileexists(filename) = false then
   begin
      exit;
   end;

   try
      writeln('Opening files...');
      assignfile(mdlfile,filename);
      assignfile(dstfile,changefileext(filename,'.md2'));
      reset(mdlfile,1);
      rewrite(dstfile,1);

      Blockread(mdlfile,Header, Sizeof(header));
      //DumpHeader(Header);
      //readln;
      if not(Header.Version = 1) then
      begin
           writeln('Wrong model version: '+inttostr(Header.Version));
           readln;
           exit;
      end;
      Dai2QuakeHeader(Header,QHeader);

      // Load Gl commands
      Getmem(GlCmds,(Header.NumGlCommands+1) * sizeof(longint));
      GetMem(QGlCmds,(Header.NumGlCommands+1) * sizeof(longint));
      fillchar(GlCmds^,(Header.NumGlCommands+1) * sizeof(longint),0);
      seek(mdlfile,Header.OffsetGlCommands);
      blockread(mdlfile,glcmds^,sizeof(longint) * header.numglcommands);
      QHeader.numGlCommands:= Dai2QuakeGlCmds(GlCmds,QGlCmds);

      //writeln('Converted gl commands');
      //readln;

      // Load the Frames (Version 1 = same as quake 2 so don't convert anything)
      Getmem(Frames,Header.NumFrames * Header.Framesize);
      seek(mdlfile,Header.OffsetFrames);
      blockread(mdlfile,Frames^,Header.Framesize * Header.NumFrames);
      GetMem(QFrames,Header.NumFrames * Header.Framesize);
      QFrames := Frames;

      //Load the triangles
      Getmem(Triangles,Header.NumTriangles * sizeof(TTriangle));
      Getmem(QTriangles,Header.NumTriangles * sizeof(TQuakeTriangle));
      seek(mdlfile,Header.OffsetTriangles);
      blockread(mdlfile,Triangles^,Header.NumTriangles * sizeof(TTriangle));
      srctri := Triangles;
      dsttri := QTriangles;
      for i:=0 to header.numTriangles-1 do
      begin
           dsttri^ := Dai2QuakeTriangle(srctri^);
           inc(cardinal(srctri),sizeof(TTriangle));
           inc(cardinal(dsttri),sizeof(TQuakeTriangle));
      end;

      //Load the texco's
      Getmem(Texcos,Header.NumTexcos * sizeof(TTextureCoordinate));
      seek(mdlfile,Header.OffsetTexcos);
      blockread(mdlfile,Texcos^,Header.NumTexCos * sizeof(TTextureCoordinate));

      //Reassemble the header offsets
      with QHeader do
      begin
           OffsetGlCommands := sizeof(QHeader);
           OffsetTriangles := OffsetGlCommands+numGlCommands*sizeof(longint);
           OffsetFrames := OffsetTriangles + (numTriangles*sizeof(TQuakeTriangle));
           OffsetTexCos := OffsetFrames+numFrames*FrameSize;
           OffsetSkins := OffsetTexCos + NumTexCos * sizeof(TTextureCoordinate);
           OffsetEnd := OffsetSkins+64;
      end;

      //Update other header data (that isn't updated yet)
      QHeader.numSkins := 1;

      //write it all out
      blockWrite(dstFile,QHeader,Sizeof(TQuake2Header));
      blockWrite(dstFile,QGlCmds^,QHeader.numGlCommands*sizeof(longint));
      blockWrite(dstFile,QTriangles^,(QHeader.numTriangles*sizeof(TQuakeTriangle)));
      blockWrite(dstFile,QFrames^,QHeader.numFrames*QHeader.FrameSize);
      blockWrite(dstFile,TexCos^,QHeader.NumTexCos * sizeof(TTextureCoordinate));
      blockWrite(dstFile,userSkin,64);

   finally
      closefile(mdlfile);
      closefile(dstfile);
   end;
   //DumpHeader(Header);
   //readln;
   //DumpHeader2(QHeader);
   //readln;
end;

(*******************************************************************************

Convert a model of version 2 to a quake model

*******************************************************************************)
procedure ConvertVer2(const filename : string);
var
   mdlfile,dstfile : File;
   Header  : TMdlHeader;
   QHeader : TQuake2Header;
   Skins   : array[0..9] of Tskinname;
   Frames2,src : pFrame2;
   i       : integer;
   point   : TVector3d;
   triangles,srctri : ^TTriangle;
   //conversion
   QHearder : TQuake2Header;
   GlCmds,QGlCmds : plongint;
   QFrames,dst : pFrame;
   TexCos : ^TTextureCoordinate;
   QTriangles,dsttri : ^TQuakeTriangle;
begin
   if fileexists(filename) = false then
   begin
     writeln('file not found');
     readln;
    exit;
    end;

   try
      writeln('Opening files');
      assignfile(mdlfile,filename);
      assignfile(dstfile,changefileext(filename,'.md2'));
      reset(mdlfile,1);
      rewrite(dstfile,1);

      Blockread(mdlfile,Header, Sizeof(header));
      //DumpHeader(Header);
      //readln;
      if not(Header.Version = 2) then
      begin
           writeln('Wrong model version: '+inttostr(Header.Version));
           readln;
           exit;
      end;
      Dai2QuakeHeader(Header,QHeader);

      writeln('Version 2: Possible loss of precision');

      // Load Gl commands
      Getmem(GlCmds,(Header.NumGlCommands+1) * sizeof(longint));
      GetMem(QGlCmds,(Header.NumGlCommands+1) * sizeof(longint));
      fillchar(GlCmds^,(Header.NumGlCommands+1) * sizeof(longint),0);
      seek(mdlfile,Header.OffsetGlCommands);
      blockread(mdlfile,glcmds^,sizeof(longint) * header.numglcommands);
      QHeader.numGlCommands:= Dai2QuakeGlCmds(GlCmds,QGlCmds);

      //writeln('Converted gl commands');
      //readln;
      // Load the Frames
      Getmem(Frames2,Header.NumFrames * Header.Framesize);
      seek(mdlfile,Header.OffsetFrames);
      blockread(mdlfile,Frames2^,Header.Framesize * Header.NumFrames);
      GetMem(QFrames,Header.NumFrames * Header.Framesize);
      src := Frames2;
      dst := QFrames;
      QHeader.FrameSize := 40+header.numVertices*4;
      for i:=0 to header.NumFrames-1 do
      begin
           Dai2QuakeFrame(src^,dst^,Header.numVertices);
           inc(cardinal(src),Header.Framesize);
           inc(cardinal(dst),QHeader.Framesize);
      end;

      //Load the triangles
      Getmem(Triangles,Header.NumTriangles * sizeof(TTriangle));
      Getmem(QTriangles,Header.NumTriangles * sizeof(TQuakeTriangle));
      seek(mdlfile,Header.OffsetTriangles);
      blockread(mdlfile,Triangles^,Header.NumTriangles * sizeof(TTriangle));
      srctri := Triangles;
      dsttri := QTriangles;
      for i:=0 to header.numTriangles-1 do
      begin
           dsttri^ := Dai2QuakeTriangle(srctri^);
           inc(cardinal(srctri),sizeof(TTriangle));
           inc(cardinal(dsttri),sizeof(TQuakeTriangle));
      end;

      //Load the texco's
      Getmem(Texcos,Header.NumTexcos * sizeof(TTextureCoordinate));
      seek(mdlfile,Header.OffsetTexcos);
      blockread(mdlfile,Texcos^,Header.NumTexCos * sizeof(TTextureCoordinate));

      //Reassemble the header offsets
      with QHeader do
      begin
           OffsetGlCommands := sizeof(QHeader);
           OffsetTriangles := OffsetGlCommands+numGlCommands*sizeof(longint);
           OffsetFrames := OffsetTriangles + (numTriangles*sizeof(TQuakeTriangle));
           OffsetTexCos := OffsetFrames+numFrames*FrameSize;
           OffsetSkins := OffsetTexCos + NumTexCos * sizeof(TTextureCoordinate);
           OffsetEnd := OffsetSkins+64;
      end;

      //Update other header data (that isn't updated yet)
      QHeader.numSkins := 1;

      //write it all out
      blockWrite(dstFile,QHeader,Sizeof(TQuake2Header));
      blockWrite(dstFile,QGlCmds^,QHeader.numGlCommands*sizeof(longint));
      blockWrite(dstFile,QTriangles^,(QHeader.numTriangles*sizeof(TQuakeTriangle)));
      blockWrite(dstFile,QFrames^,QHeader.numFrames*QHeader.FrameSize);
      blockWrite(dstFile,TexCos^,QHeader.NumTexCos * sizeof(TTextureCoordinate));
      blockWrite(dstFile,userSkin,64);

   finally
      closefile(mdlfile);
      closefile(dstfile);
   end;
   //DumpHeader(Header);
   //readln;
   //DumpHeader2(QHeader);
   //readln;
end;

//This used to be usefull to hack the model format.
procedure dumpheader(const header : TMdlHeader);
begin
with header do
     begin
     writeln('magic: '+magic);
     writeln('version: '+inttostr(version));
     writeln('origin: '+floattostr(org_x)+floattostr(org_y)+floattostr(org_z));
     writeln('frameSize: '+inttostr(frameSize));
     writeln('numSkins: '+inttostr(numSkins));
     writeln('numVertices: '+inttostr(numVertices));
     writeln('numTexCoords: '+inttostr(numTexCos));
     writeln('numTriangles: '+inttostr(numTriangles));
     writeln('numGlCommands: '+inttostr(numGlCommands));
     writeln('numFrames: '+inttostr(numFrames));
     writeln('numSurfaces: '+inttostr(numSurfaces));
     writeln('offsetSkins: '+inttostr(offsetSkins));
     writeln('offsetTexCos: '+inttostr(offsetTexCos));
     writeln('offsetTriangles: '+inttostr(offsetTriangles));
     writeln('offsetFrames: '+inttostr(offsetFrames));
     writeln('offsetGlCommands: '+inttostr(offsetGlCommands));
     writeln('offsetSurfaces: '+inttostr(offsetSurfaces));
     writeln('offsetEnd: '+inttostr(offsetEnd));
     end;
end;

//This used to be usefull to hack the model format.
procedure dumpheader2(const header : TQuake2Header);
begin
with header do
     begin
     writeln('magic: '+magic);
     writeln('version: '+inttostr(version));
     writeln('frameSize: '+inttostr(frameSize));
     writeln('numSkins: '+inttostr(numSkins));
     writeln('numVertices: '+inttostr(numVertices));
     writeln('numTexCoords: '+inttostr(numTexCos));
     writeln('numTriangles: '+inttostr(numTriangles));
     writeln('numGlCommands: '+inttostr(numGlCommands));
     writeln('numFrames: '+inttostr(numFrames));
     writeln('offsetSkins: '+inttostr(offsetSkins));
     writeln('offsetTexCos: '+inttostr(offsetTexCos));
     writeln('offsetTriangles: '+inttostr(offsetTriangles));
     writeln('offsetFrames: '+inttostr(offsetFrames));
     writeln('offsetGlCommands: '+inttostr(offsetGlCommands));
     writeln('offsetEnd: '+inttostr(offsetEnd));
     end;
end;

end.
